/************************************************************************
 * NAME:	svd15.c
 *
 * DESCR:	Support for version 1.5 of the SVD format.  You should
 *		also look at svd.[ch] and svd[0-9][0-9].[ch] which have
 *		code for the different versions of the SVD format.
 *		These were split out from this file to eliminate
 *		confusion and inadvertant changes to the old formats.
 ************************************************************************/
#include <sys/types.h>
#include <stdio.h>

#include "standard.h"
#include "floppy.h"
#include "error.h"
#include "svd.h"

/************************************************************************
 * NAME:	svd_read_15header()
 *
 * DESCR:	Reads the v1.5 header information...which is the SAME
 *		as v1.2.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
svd_read_15header(int fd, int *sectors, int *tracks)
{
    return(svd_read_12header(fd,sectors,tracks));
}

/************************************************************************
 * NAME:	svd_read_15()
 *
 * DESCR:	Read in a version 1.5 of an SVD file.  Picks up right
 *		after a version number was found.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
svd_read_15(int fd,struct floppy *floppy)
{
    int sectors, tracks;
    int side, sector, track;
    int	r;
    unsigned char blockbuf[256];
    int	 indexptr;

    if ( (r=svd_read_15header(fd,&sectors,&tracks)) != E_NONE) {
	return(r);
    }

    /* ok, now we have (sectors+1) * (tracks) of 256-byte blocks coming up	*/

    floppy->sectors = sectors;
    floppy->tracks = tracks;
    floppy->sides = 1;		/* note that this is really ignored in 1.2/1.5	*/
    floppy->encoding = UNKNOWN_ENCODING;
    floppy->write_protect = 0x00;

    /* allocate the track storage	*/

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));

    if (floppy->sides == 2) {	/* really ignored in 1.2/1.5...just for the future	*/
	floppy->side[1] = (struct track *)malloc(floppy->tracks*sizeof(struct track));
    }

    /* allocate the sectors in each track	*/
    /* note that this allocation is sometimes too big, because a given track	*/
    /* doesn't actually have that many sectors (blank ones)...but that doesn't	*/
    /* really matter, in that the whole chunk is allocated and freed		*/
    
    for (track = 0; track < floppy->tracks; track++) {
	for (side = 0; side < floppy->sides; side++) {
	    floppy->side[side][track].sector = (struct sector *)
	                         malloc(floppy->sectors*sizeof(struct sector));
	}
    }

    /* now cruise through the blocks reading in data	*/

#define THIS_SECTOR	floppy->side[side][track].sector[sector]

    for (track = 0; track < floppy->tracks; track++) {
	for (side = 0; side < floppy->sides; side++) {
	    int	blanks = 0;

	    /* first, dissect the index block for this track	*/

	    if (read(fd,blockbuf,256) != 256) {
		return(E_FATAL|E_SVD_BAD_FORMAT);
	    }

	    indexptr = 0;

	    for (sector = 0; sector < floppy->sectors; sector++) {
		int	sectortype = blockbuf[indexptr++];

		THIS_SECTOR.encoding = SVD_SECTOR_TYPE(sectortype);

		if (THIS_SECTOR.encoding == BLANK_SECTOR) {
		    /* blanks are always at the end...so if we see a blank, we're done	*/
		    /* with real data...however, we need to read in the blanks' data	*/
		    blanks++;
		}

		/* there is a "standard" amount of data per header field in the	*/
		/* track header...each format needs to advance the indexpr right.*/

		switch(THIS_SECTOR.encoding) {
		    case WD_FM:
		    case WD_MFM:
			THIS_SECTOR.id = blockbuf[indexptr++];
			THIS_SECTOR.side = blockbuf[indexptr++];
			THIS_SECTOR.sector = blockbuf[indexptr++];
			THIS_SECTOR.sizecode = blockbuf[indexptr++];
			THIS_SECTOR.headCRC = blockbuf[indexptr++];
			THIS_SECTOR.headCRC += (int)(blockbuf[indexptr++]) * 256;
			THIS_SECTOR.mark = blockbuf[indexptr++];
			THIS_SECTOR.dataCRC = blockbuf[indexptr++];
			THIS_SECTOR.dataCRC += (int)(blockbuf[indexptr++]) * 256;
			THIS_SECTOR.size = 256;
			break;
		    case H17_HSFM:
			indexptr += 19;
			if (sector == floppy->sectors - 1) {
			    indexptr += 5;		/* last sector has 5 extra		*/
			}
			THIS_SECTOR.sizecode = reflect(blockbuf[indexptr++]); /* sets the volume */
			THIS_SECTOR.id = reflect(blockbuf[indexptr++]);
			THIS_SECTOR.sector = reflect(blockbuf[indexptr++]);
			THIS_SECTOR.headCRC = reflect(blockbuf[indexptr++]);
			THIS_SECTOR.dataCRC = reflect(blockbuf[indexptr++]);
			if (THIS_SECTOR.id != 0) {
			    floppy->volume = THIS_SECTOR.sizecode;
			}
			THIS_SECTOR.size = 256;
			break;
		    case RX02:
		    case BLANK_SECTOR:
		    case UNKNOWN_ENCODING:
			indexptr += 9;
			break;
		}
	    }

	    /* NOTE that the floppy->encoding used to be devined from the type	*/
	    /* of trailer that was in the SVD file...however, now it is 	*/
	    /* generated based upon the majority sector format.			*/
	    /*									*/
	    /* floppy->encoding = SVD_TRAILER_TYPE(blockbuf[indexptr++]);	*/

	    floppy->side[side][track].sectors = floppy->sectors - blanks;

	    /* now read in the data for each of the blocks in this track	*/
	    /* note that we're reading all of the data, which includes the	*/
	    /* data for the blank tracks...which is not used however...		*/

	    /* first, align to the first data block set as appropriate for the format	*/
	    /* past the first header fields, but before the data CRC...			*/
	    /* note that since we don't yet know the overall encoding, we are counting	*/
	    /* on the fact that the first sector encoding is enough to make this work	*/

	    sector = 0;

	    switch(THIS_SECTOR.encoding) {
	        case WD_FM:
	        case WD_MFM:
		    indexptr = HEADER_FIELDS_TO_CRC15;
		    break;
	        case H17_HSFM:
		    indexptr = 24;
		    break;
		case RX02:
	        case BLANK_SECTOR:
	        case UNKNOWN_ENCODING:
		    indexptr = HEADER_FIELDS_TO_CRC15;
		    break;
	    }

	    for (sector = 0; sector < floppy->sectors; sector++) {
		int i, j;

		if (read(fd,blockbuf,256) != 256) {
		    return(E_FATAL|E_SVD_BAD_FORMAT);
		}

		j = 0;
		for (i=indexptr; i < 256; i++) {
		    switch(THIS_SECTOR.encoding) {
		        case H17_HSFM:
			    THIS_SECTOR.data[j++] = reflect(blockbuf[i]);
			    break;
		        default:
			    THIS_SECTOR.data[j++] = blockbuf[i];
			    break;
		    }
		}
		for (i = 0; i < indexptr; i++) {
		    switch(THIS_SECTOR.encoding) {
		        case H17_HSFM:
			    THIS_SECTOR.data[j++] = reflect(blockbuf[i]);
			    break;
		        default:
			    THIS_SECTOR.data[j++] = blockbuf[i];
			    break;
		    }
		}

		switch (THIS_SECTOR.encoding) {
		    case WD_FM:
		    case WD_MFM:
			indexptr += HEADER_FIELDS_15;		/* jump to next header fields	*/
			break;
		    case H17_HSFM:
			indexptr += 25;
			if (sector == floppy->sectors - 2) {
			    indexptr += 5;
			}
			break;
		    case RX02:
	            case BLANK_SECTOR:
	            case UNKNOWN_ENCODING:
			indexptr += HEADER_FIELDS_15;
		    break;
		}
	    }
	}
    }

    /* set the overall encoding */

    {
	int mixed;

	floppy->encoding = floppy_overall_encoding(floppy, &mixed, -1);
    }

    return(E_NONE);
}


/************************************************************************
 * NAME:	svd_dump_15()
 *
 * DESCR:	Dumps out the floppy information in SVD format.
 *
 * ARGS:	fd is the file descriptor to dump on
 *		do_slide is a boolean indicating whether slide is used
 *
 * RETURNS:	0 if things OK, an error number otherwise
 *
 * NOTES:	Currently assumes single density only.
 *
 *		BIG NOTE - some things aren't currently in the file:
 *			write-protect, sides, sector/floppy density (v1.2)
 ************************************************************************/
int
svd_dump_15(struct floppy *floppy, int fd)
{
    /* this will dump out the existing floppy data in SVD format	*/

    /* for each track, dump out sectors+1 with the proper rotation	*/

    int			side;
    int			track;
    int			sector;
    int			retvalue = 0;
    int			indexptr;
    unsigned char	block[256];
    unsigned char	buffer[100];

    sprintf(buffer,"%s\n","1.5");	    write(fd,buffer,strlen(buffer));
    sprintf(buffer,"%d\n",floppy->sectors); write(fd,buffer,strlen(buffer));
    sprintf(buffer,"%d\n",floppy->tracks);  write(fd,buffer,strlen(buffer));

    for (track = 0; track < floppy->tracks; track++) {

	for (side = 0; side < floppy->sides; side++) {

#define THIS_SECTOR	floppy->side[side][track].sector[sector]

/***	    if (THIS_SECTOR.density != 0 ) {
		error(E_SVD_UNSUP,E_SVD_UNSUPTEXT,"double density sector");
		return(E_SVD_UNSUP);
	    }
***/
	    { 
		int	i;

		for (i=0; i < 256; i++) {	/* ensures that an output file	*/
		    block[i] = 0;		/*   is always the same		*/
		}
	    }

	    indexptr = 0;

	    /* first compose the indexblock for the track	*/

	    for (sector = 0; sector < floppy->side[side][track].sectors; sector++) {

		block[indexptr++] = SVD_ENCODING_FLAG(THIS_SECTOR.encoding);

		switch(THIS_SECTOR.encoding) {
		    case WD_FM:
		    case WD_MFM:
			block[indexptr++] = THIS_SECTOR.id;
			block[indexptr++] = THIS_SECTOR.side;
			block[indexptr++] = THIS_SECTOR.sector;
			block[indexptr++] = THIS_SECTOR.sizecode;
			block[indexptr++] = THIS_SECTOR.headCRC & 0xff;
			block[indexptr++] = THIS_SECTOR.headCRC / 256;
			block[indexptr++] = THIS_SECTOR.mark;
			block[indexptr++] = THIS_SECTOR.dataCRC & 0xff;
			block[indexptr++] = THIS_SECTOR.dataCRC / 256;
			break;

		    case H17_HSFM:
			indexptr += 19;			/* uses pre-alignment for h17 formst	*/
			if (sector == floppy->side[side][track].sectors - 1) {
			    indexptr += 5;		/* last sector has 5 extra		*/
			}
			/* first set the right volume...always zero on first track	*/
			block[indexptr++] = reflect((THIS_SECTOR.id == 0)?0:THIS_SECTOR.sizecode);
			block[indexptr++] = reflect(THIS_SECTOR.id);
			block[indexptr++] = reflect(THIS_SECTOR.sector);
			block[indexptr++] = reflect(THIS_SECTOR.headCRC) & 0xff;
			block[indexptr++] = reflect(THIS_SECTOR.dataCRC) & 0xff;
			break;

		    case RX02:
		    case BLANK_SECTOR:
		    case UNKNOWN_ENCODING:
			indexptr += 9;		/* just blow by it	*/
			break;
		}
	    }

	    /* if the number of real sectors is lower than the "all track" sector	*/
	    /* numbers, then generate an appropriate number of blank sectors here.	*/

	    for ( ; sector < floppy->sectors; sector++) {
		block[indexptr++] = SVD_BLANK_FLAG(floppy);
		block[indexptr++] = 0x00;	/* id	-  just dummy data from here on	*/
		block[indexptr++] = 0x00;	/* side		*/
		block[indexptr++] = 0x00;	/* sector	*/
		block[indexptr++] = 0x00;	/* sizecode	*/
		block[indexptr++] = 0x00;	/* header CRC1	*/
		block[indexptr++] = 0x00;	/* header CRC2	*/
		block[indexptr++] = 0x00;	/* mark		*/
		block[indexptr++] = 0x00;	/* data CRC1	*/
		block[indexptr++] = 0x00;	/* data CRC2	*/
	    }


	    block[indexptr++] = SVD_TRAILER_FLAG(floppy);	/* type of trailer	*/

	    write(fd,block,256);

	    /* now compose and write out sectors */

	    /* first, align to the first data block set as appropriate for the format	*/
	    /* past the first header fields, but before the data CRC...			*/

	    sector = 0;

	    switch(THIS_SECTOR.encoding) {
	        case WD_FM:
	        case WD_MFM:
		    indexptr = HEADER_FIELDS_TO_CRC15;
		    break;
	        case H17_HSFM:
		    indexptr = 24;
		    break;
		case RX02:
	        case BLANK_SECTOR:
	        case UNKNOWN_ENCODING:
		    indexptr = HEADER_FIELDS_TO_CRC15;
		    break;
	    }


	    for (sector = 0; sector < floppy->side[side][track].sectors; sector++) {
		int	i, j;

		j = 0;
		for (i=indexptr; i < 256; i++) {
		    switch(THIS_SECTOR.encoding) {
		        case H17_HSFM:
			    block[i] = reflect(THIS_SECTOR.data[j++]);
			    break;
		        default:
			    block[i] = THIS_SECTOR.data[j++];
			    break;
		    }
		}
		for (i = 0; i < indexptr; i++) {
		    switch(THIS_SECTOR.encoding) {
		        case H17_HSFM:
			    block[i] = reflect(THIS_SECTOR.data[j++]);
			    break;
		        default:
			    block[i] = THIS_SECTOR.data[j++];
			    break;
		    }
		}

		write(fd,block,256);

		switch (THIS_SECTOR.encoding) {
		    case WD_FM:
		    case WD_MFM:
			indexptr += HEADER_FIELDS_15;		/* jump to next header fields	*/
			break;
		    case H17_HSFM:
			indexptr += 25;
			if (sector == floppy->side[side][track].sectors - 2) {
			    indexptr += 5;
			}
			break;
		    case RX02:
	            case BLANK_SECTOR:
	            case UNKNOWN_ENCODING:
			indexptr += HEADER_FIELDS_15;
			break;
		}

	    }

	    /* write out some dummy sectors if there are to be blanks	*/
	    /* they'll have "blank sector data" in them...note that	*/
	    /* this data isn't really "on the disk" but it is in the	*/
	    /* floppy SVD source data file				*/

	    for ( ; sector < floppy->sectors; sector++) {
		strcpy(block,"   *** blank sector data ***   ");
		write(fd,block,256);
	    }
	}
    }
    return(0);
}
